// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
// SPDX-FileCopyrightText: Copyright Contributors to the Kokkos project

#ifndef KOKKOS_TEST_TEAM_REDUCTION_SCAN_HPP
#define KOKKOS_TEST_TEAM_REDUCTION_SCAN_HPP
#include <TestTeam.hpp>

namespace Test {

TEST(TEST_CATEGORY, team_reduction_scan) {
  TestScanTeam<TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >(0);
  TestScanTeam<TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >(0);
  TestScanTeam<TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >(10);
  TestScanTeam<TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >(10);
  TestScanTeam<TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >(10000);
  TestScanTeam<TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >(10000);
}

TEST(TEST_CATEGORY, team_long_reduce) {
  {
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_test(0);
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_test(0);
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_test(3);
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_test(3);
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_test(100000);
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_test(100000);
  }
}

TEST(TEST_CATEGORY, team_double_reduce) {
  {
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_test(0);
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_test(0);
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_test(3);
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_test(3);
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_test(100000);
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_test(100000);
  }
}

TEST(TEST_CATEGORY, team_long_array_reduce) {
// FIXME_WINDOWS FIXME_32BIT Test is known to fail
#if defined(_WIN32) || defined(KOKKOS_IMPL_32BIT)
  GTEST_SKIP() << "Test known to fail on Windows or in 32-bit builds";
#endif

  {
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_array_test(0);
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_array_test(0);
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_array_test(3);
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_array_test(3);
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_array_test(100000);
    TestReduceTeam<long, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_array_test(100000);
  }
}

TEST(TEST_CATEGORY, team_double_array_reduce) {
// FIXME_MSVC FIXME_32BIT Test is known to fail
#if defined(KOKKOS_COMPILER_MSVC) || defined(KOKKOS_IMPL_32BIT)
  GTEST_SKIP() << "Test known to fail on Windows or in 32-bit builds";
#endif

  {
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_array_test(0);
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_array_test(0);
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_array_test(3);
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_array_test(3);
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Static> >{}
        .run_array_test(100000);
    TestReduceTeam<double, TEST_EXECSPACE, Kokkos::Schedule<Kokkos::Dynamic> >{}
        .run_array_test(100000);
  }
}

template <typename ExecutionSpace>
struct DummyTeamReductionFunctor {
  using TeamPolicy     = Kokkos::TeamPolicy<ExecutionSpace>;
  using TeamHandleType = typename TeamPolicy::member_type;

  KOKKOS_FUNCTION void operator()(const TeamHandleType&, double&) const {}
};

template <typename ExecutionSpace>
void test_team_parallel_reduce(const int num_loop_size) {
  using TeamPolicy = Kokkos::TeamPolicy<ExecutionSpace>;

  using ReducerType = Kokkos::Sum<double>;
  double result     = 10.;
  ReducerType reducer(result);

  const int bytes_per_team   = 0;
  const int bytes_per_thread = 117;

  TeamPolicy team_exec(num_loop_size, Kokkos::AUTO);
  team_exec.set_scratch_size(1, Kokkos::PerTeam(bytes_per_team),
                             Kokkos::PerThread(bytes_per_thread));

  Kokkos::parallel_reduce(team_exec,
                          DummyTeamReductionFunctor<ExecutionSpace>{}, reducer);
  ASSERT_EQ(result, 0.);
}

TEST(TEST_CATEGORY, team_parallel_dummy_with_reducer_and_scratch_space) {
  {
    test_team_parallel_reduce<TEST_EXECSPACE>(0);
    test_team_parallel_reduce<TEST_EXECSPACE>(1);
  }
}

TEST(TEST_CATEGORY, repeated_team_reduce) {
#ifdef KOKKOS_IMPL_32BIT
  GTEST_SKIP() << "Failing KOKKOS_IMPL_32BIT";  // FIXME_32BIT
#endif

  TestRepeatedTeamReduce<TEST_EXECSPACE>();
}

TEST(TEST_CATEGORY, nested_team_reduce_functor_as_reducer) {
  {
    TestTeamNestedReducerFunctor<TEST_EXECSPACE>().run_test_team_thread();
    TestTeamNestedReducerFunctor<TEST_EXECSPACE>().run_test_thread_vector();
    TestTeamNestedReducerFunctor<TEST_EXECSPACE>().run_test_team_vector();
  }
}

}  // namespace Test
#endif
