// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
// SPDX-FileCopyrightText: Copyright Contributors to the Kokkos project

#include <Kokkos_Macros.hpp>
#ifdef KOKKOS_ENABLE_EXPERIMENTAL_CXX20_MODULES
import kokkos.core;
#else
#include <Kokkos_Core.hpp>
#endif
#include <TestHIP_Category.hpp>

#include <array>

namespace Test {

__global__ void offset(int* p) {
  int idx = blockIdx.x * blockDim.x + threadIdx.x;
  if (idx < 100) {
    p[idx] += idx;
  }
}

// Test whether allocations survive Kokkos initialize/finalize if done via Raw
// HIP.
TEST(hip, raw_hip_interop) {
  // Make sure that we use the same device for all allocations
  Kokkos::initialize();

  int* p;
  KOKKOS_IMPL_HIP_SAFE_CALL(hipMalloc(&p, sizeof(int) * 100));

  Kokkos::View<int*, Kokkos::MemoryTraits<Kokkos::Unmanaged>> v(p, 100);
  Kokkos::deep_copy(v, 5);

  Kokkos::finalize();

  offset<<<dim3(100), dim3(100), 0, nullptr>>>(p);
  KOKKOS_IMPL_HIP_SAFE_CALL(hipDeviceSynchronize());

  std::array<int, 100> h_p;
  KOKKOS_IMPL_HIP_SAFE_CALL(
      hipMemcpy(h_p.data(), p, sizeof(int) * 100, hipMemcpyDefault));
  KOKKOS_IMPL_HIP_SAFE_CALL(hipDeviceSynchronize());
  int64_t sum        = 0;
  int64_t sum_expect = 0;
  for (int i = 0; i < 100; i++) {
    sum += h_p[i];
    sum_expect += 5 + i;
  }

  ASSERT_EQ(sum, sum_expect);
  KOKKOS_IMPL_HIP_SAFE_CALL(hipFree(p));
}
}  // namespace Test
