# Copyright Spack Project Developers. See COPYRIGHT file for details.
#
# SPDX-License-Identifier: (Apache-2.0 OR MIT)

import base64
import collections
import datetime
import email.message
import errno
import functools
import inspect
import io
import itertools
import json
import os
import re
import shutil
import stat
import sys
import tempfile
import xml.etree.ElementTree
from pathlib import Path
from typing import List, Optional, Tuple

import pytest

import spack.vendor.archspec.cpu
import spack.vendor.archspec.cpu.microarchitecture
import spack.vendor.archspec.cpu.schema

import spack.binary_distribution
import spack.bootstrap
import spack.caches
import spack.compilers.config
import spack.compilers.libraries
import spack.concretize
import spack.config
import spack.directives_meta
import spack.environment as ev
import spack.error
import spack.extensions
import spack.hash_types
import spack.llnl.util.lang
import spack.llnl.util.lock
import spack.llnl.util.tty as tty
import spack.llnl.util.tty.color
import spack.modules.common
import spack.package_base
import spack.paths
import spack.platforms
import spack.repo
import spack.solver.asp
import spack.solver.reuse
import spack.spec
import spack.stage
import spack.store
import spack.subprocess_context
import spack.tengine
import spack.util.executable
import spack.util.file_cache
import spack.util.git
import spack.util.gpg
import spack.util.naming
import spack.util.parallel
import spack.util.spack_yaml as syaml
import spack.util.url as url_util
import spack.util.web
import spack.version
from spack.enums import ConfigScopePriority
from spack.fetch_strategy import URLFetchStrategy
from spack.installer import PackageInstaller
from spack.llnl.util.filesystem import (
    copy,
    copy_tree,
    join_path,
    mkdirp,
    remove_linked_tree,
    touchp,
    working_dir,
)
from spack.main import SpackCommand
from spack.util.pattern import Bunch
from spack.util.remote_file_cache import raw_github_gitlab_url

mirror_cmd = SpackCommand("mirror")


def _recursive_chmod(path: Path, mode: int):
    """Recursively change permissions of a directory and all its contents."""
    path.chmod(mode)
    for root, dirs, files in os.walk(path):
        for file in files:
            os.chmod(os.path.join(root, file), mode)
        for dir in dirs:
            os.chmod(os.path.join(root, dir), mode)


@pytest.fixture(autouse=True)
def clear_sys_modules():
    """Clear package repos from sys.modules before each test."""
    for key in list(sys.modules.keys()):
        if key.startswith("spack_repo.") or key == "spack_repo":
            del sys.modules[key]
    yield


@pytest.fixture(autouse=True)
def check_config_fixture(request):
    if "config" in request.fixturenames and "mutable_config" in request.fixturenames:
        raise RuntimeError("'config' and 'mutable_config' are both requested")


def ensure_configuration_fixture_run_before(request):
    """Ensure that fixture mutating the configuration run before the one where
    the function is called.
    """
    if "config" in request.fixturenames:
        request.getfixturevalue("config")
    if "mutable_config" in request.fixturenames:
        request.getfixturevalue("mutable_config")


@pytest.fixture(scope="session")
def git():
    """Fixture for tests that use git."""
    try:
        return spack.util.git.git(required=True)
    except spack.util.executable.CommandNotFoundError:
        pytest.skip("requires git to be installed")


#
# Return list of shas for latest two git commits in local spack repo
#
@pytest.fixture(scope="session")
def last_two_git_commits(git):
    spack_git_path = spack.paths.prefix
    with working_dir(spack_git_path):
        git_log_out = git("log", "-n", "2", output=str, error=os.devnull)

    regex = re.compile(r"^commit\s([^\s]+$)", re.MULTILINE)
    yield regex.findall(git_log_out)


def write_file(filename, contents):
    with open(filename, "w", encoding="utf-8") as f:
        f.write(contents)


commit_counter = 0


@pytest.fixture
def override_git_repos_cache_path(tmp_path: Path):
    saved = spack.paths.user_repos_cache_path
    tmp_git_path = tmp_path / "git-repo-cache-path-for-tests"
    tmp_git_path.mkdir()
    spack.paths.user_repos_cache_path = str(tmp_git_path)
    yield
    spack.paths.user_repos_cache_path = saved


@pytest.fixture
def mock_git_version_info(git, tmp_path: Path, override_git_repos_cache_path):
    """Create a mock git repo with known structure

    The structure of commits in this repo is as follows::

       | o fourth 1.x commit (1.2)
       | o third 1.x commit
       | |
       o | fourth main commit (v2.0)
       o | third main commit
       | |
       | o second 1.x commit (v1.1)
       | o first 1.x commit
       | /
       |/
       o second commit (v1.0)
       o first commit

    The repo consists of a single file, in which the GitVersion.std_version representation
    of each commit is expressed as a string.

    Important attributes of the repo for test coverage are: multiple branches,
    version tags on multiple branches, and version order is not equal to time
    order or topological order.
    """
    repo_dir = tmp_path / "git_version_info_repo"
    repo_dir.mkdir()
    repo_path = str(repo_dir)
    filename = "file.txt"

    def commit(message):
        global commit_counter
        git(
            "commit",
            "--no-gpg-sign",
            "--date",
            "2020-01-%02d 12:0:00 +0300" % commit_counter,
            "-am",
            message,
        )
        commit_counter += 1

    with working_dir(repo_path):
        git("init")

        git("config", "user.name", "Spack")
        git("config", "user.email", "spack@spack.io")
        git("checkout", "-b", "main")

        commits = []

        def latest_commit():
            return git("rev-list", "-n1", "HEAD", output=str, error=str).strip()

        # Add two commits on main branch
        # A commit without a previous version counts as "0"
        write_file(filename, "[0]")
        git("add", filename)
        commit("first commit")
        commits.append(latest_commit())

        # Tag second commit as v1.0
        write_file(filename, "[1, 0]")
        commit("second commit")
        commits.append(latest_commit())
        git("tag", "v1.0")

        # Add two commits and a tag on 1.x branch
        git("checkout", "-b", "1.x")
        write_file(filename, "[1, 0, 'git', 1]")
        commit("first 1.x commit")
        commits.append(latest_commit())

        write_file(filename, "[1, 1]")
        commit("second 1.x commit")
        commits.append(latest_commit())
        git("tag", "v1.1")

        # Add two commits and a tag on main branch
        git("checkout", "main")
        write_file(filename, "[1, 0, 'git', 1]")
        commit("third main commit")
        commits.append(latest_commit())
        write_file(filename, "[2, 0]")
        commit("fourth main commit")
        commits.append(latest_commit())
        git("tag", "v2.0")

        # Add two more commits on 1.x branch to ensure we aren't cheating by using time
        git("checkout", "1.x")
        write_file(filename, "[1, 1, 'git', 1]")
        commit("third 1.x commit")
        commits.append(latest_commit())
        write_file(filename, "[1, 2]")
        commit("fourth 1.x commit")
        commits.append(latest_commit())
        git("tag", "1.2")  # test robust parsing to different syntax, no v

        # The commits are ordered with the last commit first in the list
        commits = list(reversed(commits))

    # Return the git directory to install, the filename used, and the commits
    yield repo_path, filename, commits


@pytest.fixture
def mock_git_package_changes(git, tmp_path: Path, override_git_repos_cache_path, monkeypatch):
    """Create a mock git repo with known structure of package edits

    The structure of commits in this repo is as follows::

       o diff-test: add v1.2 (from a git ref)
       |
       o diff-test: add v1.1 (from source tarball)
       |
       o diff-test: new package (testing multiple added versions)

    The repo consists of a single package.py file for DiffTest.

    Important attributes of the repo for test coverage are: multiple package
    versions are added with some coming from a tarball and some from git refs.
    """
    filename = "diff_test/package.py"

    repo_path, _ = spack.repo.create_repo(str(tmp_path), namespace="myrepo")
    cache_dir = tmp_path / "cache"
    cache_dir.mkdir()
    repo_cache = spack.util.file_cache.FileCache(str(cache_dir))

    repo = spack.repo.Repo(repo_path, cache=repo_cache)

    def commit(message):
        global commit_counter
        git(
            "commit",
            "--no-gpg-sign",
            "--date",
            "2020-01-%02d 12:0:00 +0300" % commit_counter,
            "-am",
            message,
        )
        commit_counter += 1

    with working_dir(repo.packages_path):
        git("init")

        git("config", "user.name", "Spack")
        git("config", "user.email", "spack@spack.io")

        commits = []

        def latest_commit():
            return git("rev-list", "-n1", "HEAD", output=str, error=str).strip()

        os.makedirs(os.path.dirname(filename))

        # add pkg-a as a new package to the repository
        shutil.copy2(f"{spack.paths.test_path}/data/conftest/diff-test/package-0.txt", filename)
        git("add", filename)
        commit("diff-test: new package")
        commits.append(latest_commit())

        # add v2.1.5 to pkg-a
        shutil.copy2(f"{spack.paths.test_path}/data/conftest/diff-test/package-1.txt", filename)
        git("add", filename)
        commit("diff-test: add v2.1.5")
        commits.append(latest_commit())

        # add v2.1.6 to pkg-a
        shutil.copy2(f"{spack.paths.test_path}/data/conftest/diff-test/package-2.txt", filename)
        git("add", filename)
        commit("diff-test: add v2.1.6")
        commits.append(latest_commit())

        # The commits are ordered with the last commit first in the list
        commits = list(reversed(commits))

    # Return the git directory to install, the filename used, and the commits
    yield repo, filename, commits


@pytest.fixture(autouse=True)
def clear_recorded_monkeypatches():
    yield
    spack.subprocess_context.MONKEYPATCHES.clear()


@pytest.fixture(scope="session", autouse=True)
def record_monkeypatch_setattr():
    import _pytest

    saved_setattr = _pytest.monkeypatch.MonkeyPatch.setattr

    def record_setattr(cls, target, name, value, *args, **kwargs):
        spack.subprocess_context.MONKEYPATCHES.append((target, name))
        saved_setattr(cls, target, name, value, *args, **kwargs)

    _pytest.monkeypatch.MonkeyPatch.setattr = record_setattr
    try:
        yield
    finally:
        _pytest.monkeypatch.MonkeyPatch.setattr = saved_setattr


def _can_access(path, perms):
    return False


@pytest.fixture
def no_path_access(monkeypatch):
    monkeypatch.setattr(os, "access", _can_access)


#
# Disable any active Spack environment BEFORE all tests
#
@pytest.fixture(scope="session", autouse=True)
def clean_user_environment():
    spack_env_value = os.environ.pop(ev.spack_env_var, None)
    with ev.no_active_environment():
        yield
    if spack_env_value:
        os.environ[ev.spack_env_var] = spack_env_value


#
# Make sure global state of active env does not leak between tests.
#
@pytest.fixture(scope="function", autouse=True)
def clean_test_environment():
    yield
    ev.deactivate()


def _host():
    """Mock archspec host so there is no inconsistency on the Windows platform
    This function cannot be local as it needs to be pickleable"""
    return spack.vendor.archspec.cpu.Microarchitecture("x86_64", [], "generic", [], {}, 0)


@pytest.fixture(scope="function")
def archspec_host_is_spack_test_host(monkeypatch):
    monkeypatch.setattr(spack.vendor.archspec.cpu, "host", _host)


# Hooks to add command line options or set other custom behaviors.
# They must be placed here to be found by pytest. See:
#
# https://docs.pytest.org/en/latest/writing_plugins.html
#
def pytest_addoption(parser):
    group = parser.getgroup("Spack specific command line options")
    group.addoption(
        "--fast",
        action="store_true",
        default=False,
        help='runs only "fast" unit tests, instead of the whole suite',
    )


def pytest_collection_modifyitems(config, items):
    if not config.getoption("--fast"):
        # --fast not given, run all the tests
        return

    slow_tests = ["db", "network", "maybeslow"]
    skip_as_slow = pytest.mark.skip(reason="skipped slow test [--fast command line option given]")
    for item in items:
        if any(x in item.keywords for x in slow_tests):
            item.add_marker(skip_as_slow)


@pytest.fixture(scope="function")
def use_concretization_cache(mock_packages, mutable_config, tmp_path: Path):
    """Enables the use of the concretization cache"""
    conc_cache_dir = tmp_path / "concretization"
    conc_cache_dir.mkdir()

    # ensure we have an isolated concretization cache while using fixture
    with spack.config.override(
        "concretizer:concretization_cache", {"enable": True, "url": str(conc_cache_dir)}
    ):
        yield conc_cache_dir


#
# These fixtures are applied to all tests
#
@pytest.fixture(scope="function", autouse=True)
def no_chdir():
    """Ensure that no test changes Spack's working dirctory.

    This prevents Spack tests (and therefore Spack commands) from
    changing the working directory and causing other tests to fail
    mysteriously. Tests should use ``working_dir`` or ``py.path``'s
    ``.as_cwd()`` instead of ``os.chdir`` to avoid failing this check.

    We assert that the working directory hasn't changed, unless the
    original wd somehow ceased to exist.

    """
    original_wd = os.getcwd()
    yield
    if os.path.isdir(original_wd):
        assert os.getcwd() == original_wd


def onerror(func, path, error_info):
    # Python on Windows is unable to remvove paths without
    # write (IWUSR) permissions (such as those generated by Git on Windows)
    # This method changes file permissions to allow removal by Python
    os.chmod(path, stat.S_IWUSR)
    func(path)


@pytest.fixture(scope="function", autouse=True)
def mock_stage(tmp_path_factory: pytest.TempPathFactory, monkeypatch, request):
    """Establish the temporary build_stage for the mock archive."""
    # The approach with this autouse fixture is to set the stage root
    # instead of using spack.config.override() to avoid configuration
    # conflicts with dozens of tests that rely on other configuration
    # fixtures, such as config.

    if "nomockstage" in request.keywords:
        # Tests can opt-out with @pytest.mark.nomockstage
        yield None
        return

    # Set the build stage to the requested path
    new_stage = tmp_path_factory.mktemp("mock-stage")

    # Ensure the source directory exists within the new stage path
    source_path = new_stage / spack.stage._source_path_subdir
    source_path.mkdir(parents=True, exist_ok=True)

    monkeypatch.setattr(spack.stage, "_stage_root", str(new_stage))

    yield str(new_stage)

    # Clean up the test stage directory
    if new_stage.is_dir():
        shutil.rmtree(new_stage, onerror=onerror)


@pytest.fixture(scope="session")
def mock_stage_for_database(tmp_path_factory: pytest.TempPathFactory, monkeypatch_session):
    """A session-scoped analog of mock_stage, so that the mock_store
    fixture uses its own stage vs. the global stage root for spack.
    """
    new_stage = tmp_path_factory.mktemp("mock-stage")

    source_path = new_stage / spack.stage._source_path_subdir
    source_path.mkdir(parents=True, exist_ok=True)

    monkeypatch_session.setattr(spack.stage, "_stage_root", str(new_stage))

    yield str(new_stage)

    # Clean up the test stage directory
    if new_stage.is_dir():
        shutil.rmtree(new_stage, onerror=onerror)


@pytest.fixture(scope="session")
def ignore_stage_files():
    """Session-scoped helper for check_for_leftover_stage_files.

    Used to track which leftover files in the stage have been seen.
    """
    # to start with, ignore the .lock file at the stage root.
    return set([".lock", spack.stage._source_path_subdir, "build_cache"])


def remove_whatever_it_is(path):
    """Type-agnostic remove."""
    if os.path.isfile(path):
        os.remove(path)
    elif os.path.islink(path):
        remove_linked_tree(path)
    else:
        shutil.rmtree(path, onerror=onerror)


@pytest.fixture
def working_env():
    saved_env = os.environ.copy()
    yield
    # os.environ = saved_env doesn't work
    # it causes module_parsing::test_module_function to fail
    # when it's run after any test using this fixutre
    os.environ.clear()
    os.environ.update(saved_env)


@pytest.fixture(scope="function", autouse=True)
def check_for_leftover_stage_files(request, mock_stage, ignore_stage_files):
    """
    Ensure that each (mock_stage) test leaves a clean stage when done.

    Tests that are expected to dirty the stage can disable the check by
    adding::

        @pytest.mark.disable_clean_stage_check

    and the associated stage files will be removed.
    """
    yield

    if mock_stage is None:
        # When tests opt out with @pytest.mark.nomockstage, do not check for left-over files
        return

    files_in_stage = set()
    try:
        stage_files = os.listdir(mock_stage)
        files_in_stage = set(stage_files) - ignore_stage_files
    except OSError as err:
        if err.errno == errno.ENOENT or err.errno == errno.EINVAL:
            pass
        else:
            raise

    if "disable_clean_stage_check" in request.keywords:
        # clean up after tests that are expected to be dirty
        for f in files_in_stage:
            path = os.path.join(mock_stage, f)
            remove_whatever_it_is(path)
    else:
        ignore_stage_files |= files_in_stage
        assert not files_in_stage


class MockCache:
    def store(self, copy_cmd, relative_dest):
        pass

    def fetcher(self, target_path, digest, **kwargs):
        return MockCacheFetcher()


class MockCacheFetcher:
    def fetch(self):
        raise spack.error.FetchError("Mock cache always fails for tests")

    def __str__(self):
        return "[mock fetch cache]"


@pytest.fixture(autouse=True)
def mock_fetch_cache(monkeypatch):
    """Substitutes spack.paths.FETCH_CACHE with a mock object that does nothing
    and raises on fetch.
    """
    monkeypatch.setattr(spack.caches, "FETCH_CACHE", MockCache())


@pytest.fixture()
def mock_binary_index(monkeypatch, tmp_path_factory: pytest.TempPathFactory):
    """Changes the directory for the binary index and creates binary index for
    every test. Clears its own index when it's done.
    """
    tmpdir = tmp_path_factory.mktemp("mock_binary_index")
    index_path = tmpdir / "binary_index"
    mock_index = spack.binary_distribution.BinaryCacheIndex(str(index_path))
    monkeypatch.setattr(spack.binary_distribution, "BINARY_INDEX", mock_index)
    yield


@pytest.fixture(autouse=True)
def _skip_if_missing_executables(request, monkeypatch):
    """Permits to mark tests with 'require_executables' and skip the
    tests if the executables passed as arguments are not found.
    """
    marker = request.node.get_closest_marker("requires_executables")
    if marker:
        required_execs = marker.args
        missing_execs = [x for x in required_execs if spack.util.executable.which(x) is None]
        if missing_execs:
            msg = "could not find executables: {0}"
            pytest.skip(msg.format(", ".join(missing_execs)))

        # In case we require a compiler, clear the caches used to speed-up detection
        monkeypatch.setattr(spack.compilers.libraries.DefaultDynamicLinkerFilter, "_CACHE", {})


@pytest.fixture(scope="session")
def test_platform():
    return spack.platforms.Test()


@pytest.fixture(autouse=True, scope="session")
def _use_test_platform(test_platform):
    # This is the only context manager used at session scope (see note
    # below for more insight) since we want to use the test platform as
    # a default during tests.
    with spack.platforms.use_platform(test_platform):
        yield


#
# Note on context managers used by fixtures
#
# Because these context managers modify global state, they should really
# ONLY be used persistently (i.e., around yield statements) in
# function-scoped fixtures, OR in autouse session- or module-scoped
# fixtures.
#
# If they're used in regular tests or in module-scoped fixtures that are
# then injected as function arguments, weird things can happen, because
# the original state won't be restored until *after* the fixture is
# destroyed.  This makes sense for an autouse fixture, where you know
# everything in the module/session is going to need the modified
# behavior, but modifying global state for one function in a way that
# won't be restored until after the module or session is done essentially
# leaves garbage behind for other tests.
#
# In general, we should module- or session-scope the *STATE* required for
# these global objects, but we shouldn't module- or session-scope their
# *USE*, or things can get really confusing.
#


#
# Test-specific fixtures
#
@pytest.fixture(scope="session")
def mock_packages_repo():
    yield spack.repo.from_path(spack.paths.mock_packages_path)


def _pkg_install_fn(pkg, spec, prefix):
    # sanity_check_prefix requires something in the install directory
    mkdirp(prefix.bin)
    if not os.path.exists(spec.package.install_log_path):
        touchp(spec.package.install_log_path)


@pytest.fixture
def mock_pkg_install(monkeypatch):
    monkeypatch.setattr(spack.package_base.PackageBase, "install", _pkg_install_fn, raising=False)


@pytest.fixture(scope="function")
def fake_db_install(tmp_path):
    """This fakes "enough" of the installation process to make Spack
    think of a spec as being installed as far as the concretizer
    and parser are concerned. It does not run any build phase defined
    in the package, simply acting as though the installation had
    completed successfully.

    It allows doing things like

    ``spack.concretize.concretize_one(f"x ^/hash-of-y")``

    after doing something like ``fake_db_install(y)``
    """
    with spack.store.use_store(str(tmp_path)) as the_store:

        def _install(a_spec):
            the_store.db.add(a_spec)

        yield _install


@pytest.fixture(scope="function")
def mock_packages(mock_packages_repo, mock_pkg_install, request):
    """Use the 'builtin_mock' repository instead of 'builtin'"""
    ensure_configuration_fixture_run_before(request)
    with spack.repo.use_repositories(mock_packages_repo) as mock_repo:
        yield mock_repo


@pytest.fixture(scope="function")
def mutable_mock_repo(mock_packages_repo, request):
    """Function-scoped mock packages, for tests that need to modify them."""
    ensure_configuration_fixture_run_before(request)
    mock_repo = spack.repo.from_path(spack.paths.mock_packages_path)
    with spack.repo.use_repositories(mock_repo) as mock_packages_repo:
        yield mock_packages_repo


class RepoBuilder:
    """Build a mock repository in a directory"""

    _counter = 0

    def __init__(self, root_directory: str) -> None:
        RepoBuilder._counter += 1
        namespace = f"test_namespace_{RepoBuilder._counter}"
        repo_root = os.path.join(root_directory, namespace)
        os.makedirs(repo_root, exist_ok=True)
        self.template_dirs = (os.path.join(spack.paths.share_path, "templates"),)
        self.root, self.namespace = spack.repo.create_repo(repo_root, namespace)
        self.build_system_name = f"test_build_system_{self.namespace}"
        self._add_build_system()

    def add_package(
        self,
        name: str,
        dependencies: Optional[List[Tuple[str, Optional[str], Optional[str]]]] = None,
    ) -> None:
        """Create a mock package in the repository, using a Jinja2 template.

        Args:
            name: name of the new package
            dependencies: list of ("dep_spec", "dep_type", "condition") tuples.
                Both "dep_type" and "condition" can default to ``None`` in which case
                ``spack.dependency.default_deptype`` and ``spack.spec.Spec()`` are used.
        """
        dependencies = dependencies or []
        context = {
            "cls_name": spack.util.naming.pkg_name_to_class_name(name),
            "dependencies": dependencies,
        }
        template = spack.tengine.make_environment_from_dirs(self.template_dirs).get_template(
            "mock-repository/package.pyt"
        )
        package_py = self._recipe_filename(name)
        os.makedirs(os.path.dirname(package_py), exist_ok=True)
        with open(package_py, "w", encoding="utf-8") as f:
            f.write(template.render(context))

    def remove(self, name: str) -> None:
        package_py = self._recipe_filename(name)
        shutil.rmtree(os.path.dirname(package_py))

    def _add_build_system(self) -> None:
        """Add spack_repo.<namespace>.build_systems.test_build_system with
        build_system=test_build_system_<namespace>."""
        template = spack.tengine.make_environment_from_dirs(self.template_dirs).get_template(
            "mock-repository/build_system.pyt"
        )
        text = template.render({"build_system_name": self.build_system_name})
        build_system_py = os.path.join(self.root, "build_systems", "test_build_system.py")
        os.makedirs(os.path.dirname(build_system_py), exist_ok=True)
        with open(build_system_py, "w", encoding="utf-8") as f:
            f.write(text)

    def _recipe_filename(self, name: str) -> str:
        return os.path.join(
            self.root,
            "packages",
            spack.util.naming.pkg_name_to_pkg_dir(name, package_api=(2, 0)),
            "package.py",
        )


@pytest.fixture
def repo_builder(tmp_path: Path):
    return RepoBuilder(str(tmp_path))


@pytest.fixture()
def mock_custom_repository(tmp_path: Path, mutable_mock_repo):
    """Create a custom repository with a single package "c" and return its path."""
    builder = RepoBuilder(str(tmp_path))
    builder.add_package("pkg-c")
    return builder.root


@pytest.fixture(scope="session")
def linux_os():
    """Returns a named tuple with attributes 'name' and 'version'
    representing the OS.
    """
    platform = spack.platforms.host()
    name, version = "debian", "6"
    if platform.name == "linux":
        current_os = platform.default_operating_system()
        name, version = current_os.name, current_os.version
    LinuxOS = collections.namedtuple("LinuxOS", ["name", "version"])
    return LinuxOS(name=name, version=version)


@pytest.fixture
def ensure_debug(monkeypatch):
    current_debug_level = tty.debug_level()
    tty.set_debug(1)

    yield

    tty.set_debug(current_debug_level)


@pytest.fixture
def default_config():
    """Isolates the default configuration from the user configs.

    This ensures we can test the real default configuration without having
    tests fail when the user overrides the defaults that we test against."""
    defaults_path = os.path.join(spack.paths.etc_path, "defaults")
    if sys.platform == "win32":
        defaults_path = os.path.join(defaults_path, "windows")
    with spack.config.use_configuration(defaults_path) as defaults_config:
        yield defaults_config


@pytest.fixture(scope="session")
def mock_uarch_json(tmp_path_factory: pytest.TempPathFactory):
    """Mock microarchitectures.json with test architecture descriptions."""
    tmpdir = tmp_path_factory.mktemp("microarchitectures")

    uarch_json_source = (
        Path(spack.paths.test_path) / "data" / "microarchitectures" / "microarchitectures.json"
    )
    uarch_json_dest = tmpdir / "microarchitectures.json"
    shutil.copy2(uarch_json_source, uarch_json_dest)
    yield str(uarch_json_dest)


@pytest.fixture(scope="session")
def mock_uarch_configuration(mock_uarch_json):
    """Create mock dictionaries for the spack.vendor.archspec.cpu."""

    def load_json():
        with open(mock_uarch_json, encoding="utf-8") as f:
            return json.load(f)

    targets_json = load_json()
    targets = spack.vendor.archspec.cpu.microarchitecture._known_microarchitectures()

    yield targets_json, targets


@pytest.fixture(scope="function")
def mock_targets(mock_uarch_configuration, monkeypatch):
    """Use this fixture to enable mock uarch targets for testing."""
    targets_json, targets = mock_uarch_configuration
    monkeypatch.setattr(spack.vendor.archspec.cpu.schema, "TARGETS_JSON", targets_json)
    monkeypatch.setattr(spack.vendor.archspec.cpu.microarchitecture, "TARGETS", targets)


@pytest.fixture(scope="session")
def configuration_dir(tmp_path_factory: pytest.TempPathFactory, linux_os):
    """Copies mock configuration files in a temporary directory. Returns the
    directory path.
    """
    tmp_path = tmp_path_factory.mktemp("configurations")
    install_tree_root = tmp_path_factory.mktemp("opt")
    modules_root = tmp_path_factory.mktemp("share")
    tcl_root = modules_root / "modules"
    tcl_root.mkdir()
    lmod_root = modules_root / "lmod"
    lmod_root.mkdir()

    # <test_path>/data/config has mock config yaml files in it
    # copy these to the site config.
    test_config = Path(spack.paths.test_path) / "data" / "config"
    shutil.copytree(test_config, tmp_path / "site")

    # Create temporary 'defaults', 'site' and 'user' folders
    (tmp_path / "user").mkdir()

    # Fill out config.yaml, compilers.yaml and modules.yaml templates.
    locks = sys.platform != "win32"
    config = tmp_path / "site" / "config.yaml"
    config_template = test_config / "config.yaml"
    config.write_text(config_template.read_text().format(install_tree_root, locks))

    target = str(spack.vendor.archspec.cpu.host().family)
    compilers = tmp_path / "site" / "packages.yaml"
    compilers_template = test_config / "packages.yaml"
    compilers.write_text(compilers_template.read_text().format(linux_os=linux_os, target=target))

    modules = tmp_path / "site" / "modules.yaml"
    modules_template = test_config / "modules.yaml"
    modules.write_text(modules_template.read_text().format(tcl_root, lmod_root))
    yield tmp_path


def _create_mock_configuration_scopes(configuration_dir):
    """Create the configuration scopes used in `config` and `mutable_config`."""
    return [
        (
            ConfigScopePriority.DEFAULTS,
            spack.config.InternalConfigScope("_builtin", spack.config.CONFIG_DEFAULTS),
        ),
        (
            ConfigScopePriority.CONFIG_FILES,
            spack.config.DirectoryConfigScope("site", str(configuration_dir / "site")),
        ),
        (
            ConfigScopePriority.CONFIG_FILES,
            spack.config.DirectoryConfigScope("system", str(configuration_dir / "system")),
        ),
        (
            ConfigScopePriority.CONFIG_FILES,
            spack.config.DirectoryConfigScope("user", str(configuration_dir / "user")),
        ),
        (ConfigScopePriority.COMMAND_LINE, spack.config.InternalConfigScope("command_line")),
    ]


@pytest.fixture(scope="session")
def mock_configuration_scopes(configuration_dir):
    """Create a persistent Configuration object from the configuration_dir."""
    yield _create_mock_configuration_scopes(configuration_dir)


@pytest.fixture(scope="function")
def config(mock_configuration_scopes):
    """This fixture activates/deactivates the mock configuration."""
    with spack.config.use_configuration(*mock_configuration_scopes) as config:
        yield config


@pytest.fixture(scope="function")
def mutable_config(tmp_path_factory: pytest.TempPathFactory, configuration_dir):
    """Like config, but tests can modify the configuration."""
    mutable_dir = tmp_path_factory.mktemp("mutable_config") / "tmp"
    shutil.copytree(configuration_dir, mutable_dir)

    scopes = _create_mock_configuration_scopes(mutable_dir)
    with spack.config.use_configuration(*scopes) as cfg:
        yield cfg


@pytest.fixture(scope="function")
def mutable_empty_config(tmp_path_factory: pytest.TempPathFactory, configuration_dir):
    """Empty configuration that can be modified by the tests."""
    mutable_dir = tmp_path_factory.mktemp("mutable_config") / "tmp"
    scopes = [
        spack.config.DirectoryConfigScope(name, str(mutable_dir / name))
        for name in ["site", "system", "user"]
    ]

    with spack.config.use_configuration(*scopes) as cfg:
        yield cfg


# From  https://github.com/pytest-dev/pytest/issues/363#issuecomment-1335631998
# Current suggested implementation from issue compatible with pytest >= 6.2
# this may be subject to change as new versions of Pytest are released
# and update the suggested solution
@pytest.fixture(scope="session")
def monkeypatch_session():
    with pytest.MonkeyPatch.context() as monkeypatch:
        yield monkeypatch


@pytest.fixture(autouse=True)
def mock_wsdk_externals(monkeypatch):
    """Skip check for required external packages on Windows during testing."""
    monkeypatch.setattr(spack.bootstrap, "ensure_winsdk_external_or_raise", _return_none)


@pytest.fixture(scope="function")
def concretize_scope(mutable_config, tmp_path: Path):
    """Adds a scope for concretization preferences"""
    concretize_dir = tmp_path / "concretize"
    concretize_dir.mkdir()
    with spack.config.override(
        spack.config.DirectoryConfigScope("concretize", str(concretize_dir))
    ):
        yield str(concretize_dir)

    spack.repo.PATH._provider_index = None


@pytest.fixture
def no_packages_yaml(mutable_config):
    """Creates a temporary configuration without compilers.yaml"""
    for local_config in mutable_config.scopes.values():
        if not isinstance(local_config, spack.config.DirectoryConfigScope):
            continue
        compilers_yaml = local_config.get_section_filename("packages")
        if os.path.exists(compilers_yaml):
            os.remove(compilers_yaml)
    mutable_config.clear_caches()
    return mutable_config


@pytest.fixture()
def mock_low_high_config(tmp_path: Path):
    """Mocks two configuration scopes: 'low' and 'high'."""
    scopes = [
        spack.config.DirectoryConfigScope(name, str(tmp_path / name)) for name in ["low", "high"]
    ]

    with spack.config.use_configuration(*scopes) as config:
        yield config


def create_config_scope(path: Path, name: str) -> spack.config.DirectoryConfigScope:
    """helper for creating config scopes with included file/directory scopes
    that do not have existing representation on the filesystem"""
    base_scope_dir = path / "base"
    config_data = syaml.syaml_dict(
        {
            "include": [
                {
                    "name": "sub_base",
                    "path": str(path / name),
                    "optional": True,
                    "prefer_modify": True,
                }
            ]
        }
    )
    base_scope_dir.mkdir()
    with open(str(base_scope_dir / "include.yaml"), "w+", encoding="utf-8") as f:
        syaml.dump_config(config_data, stream=f, default_flow_style=False)
    scope = spack.config.DirectoryConfigScope("base", str(base_scope_dir))
    return scope


@pytest.fixture()
def mock_missing_dir_include_scopes(tmp_path: Path):
    """Mocks a config scope containing optional directory scope
    includes that do not have represetation on the filesystem"""
    scope = create_config_scope(tmp_path, "sub")

    with spack.config.use_configuration(scope) as config:
        yield config


@pytest.fixture
def mock_missing_file_include_scopes(tmp_path: Path):
    """Mocks a config scope containing optional file scope
    includes that do not have represetation on the filesystem"""
    scope = create_config_scope(tmp_path, "sub.yaml")

    with spack.config.use_configuration(scope) as config:
        yield config


def _populate(mock_db):
    r"""Populate a mock database with packages.

    Here is what the mock DB looks like (explicit roots at top):

    o  mpileaks     o  mpileaks'    o  mpileaks''     o externaltest     o trivial-smoke-test
    |\              |\              |\                |
    | o  callpath   | o  callpath'  | o  callpath''   o externaltool
    |/|             |/|             |/|               |
    o |  mpich      o |  mpich2     o |  zmpi         o externalvirtual
      |               |             o |  fake
      |               |               |
      |               |______________/
      | .____________/
      |/
      o  dyninst
      |\
      | o  libdwarf
      |/
      o  libelf
    """

    def _install(spec):
        s = spack.concretize.concretize_one(spec)
        PackageInstaller([s.package], fake=True, explicit=True).install()

    _install("mpileaks ^mpich")
    _install("mpileaks ^mpich2")
    _install("mpileaks ^zmpi")
    _install("externaltest ^externalvirtual")
    _install("trivial-smoke-test")


@pytest.fixture(scope="session")
def _store_dir_and_cache(tmp_path_factory: pytest.TempPathFactory):
    """Returns the directory where to build the mock database and
    where to cache it.
    """
    store = tmp_path_factory.mktemp("mock_store")
    cache = tmp_path_factory.mktemp("mock_store_cache")
    return store, cache


@pytest.fixture(scope="session")
def mock_store(
    tmp_path_factory: pytest.TempPathFactory,
    mock_packages_repo,
    mock_configuration_scopes,
    _store_dir_and_cache: Tuple[Path, Path],
    mock_stage_for_database,
):
    """Creates a read-only mock database with some packages installed note
    that the ref count for dyninst here will be 3, as it's recycled
    across each install.

    This does not actually activate the store for use by Spack -- see the
    ``database`` fixture for that.

    """
    store_path, store_cache = _store_dir_and_cache
    _mock_wsdk_externals = spack.bootstrap.ensure_winsdk_external_or_raise

    # Make the DB filesystem read-only to ensure constructors don't modify anything in it.
    # We want Spack to be able to point to a DB on a read-only filesystem easily.
    _recursive_chmod(store_path, 0o555)

    # If the cache does not exist populate the store and create it
    if not os.path.exists(str(store_cache / ".spack-db")):
        with spack.config.use_configuration(*mock_configuration_scopes):
            with spack.store.use_store(str(store_path)) as store:
                with spack.repo.use_repositories(mock_packages_repo):
                    # make the DB filesystem writable only while we populate it
                    _recursive_chmod(store_path, 0o755)
                    try:
                        spack.bootstrap.ensure_winsdk_external_or_raise = _return_none
                        _populate(store.db)
                    finally:
                        spack.bootstrap.ensure_winsdk_external_or_raise = _mock_wsdk_externals
                    _recursive_chmod(store_path, 0o555)

        _recursive_chmod(store_cache, 0o755)
        copy_tree(str(store_path), str(store_cache))
        _recursive_chmod(store_cache, 0o555)

    yield store_path


@pytest.fixture(scope="function")
def database(mock_store, mock_packages, config):
    """This activates the mock store, packages, AND config."""
    with spack.store.use_store(str(mock_store)) as store:
        yield store.db
        # Force reading the database again between tests
        store.db.last_seen_verifier = ""


@pytest.fixture(scope="function")
def database_mutable_config(mock_store, mock_packages, mutable_config, monkeypatch):
    """This activates the mock store, packages, AND config."""
    with spack.store.use_store(str(mock_store)) as store:
        yield store.db
        store.db.last_seen_verifier = ""


@pytest.fixture(scope="function")
def mutable_database(database_mutable_config, _store_dir_and_cache: Tuple[Path, Path]):
    """Writeable version of the fixture, restored to its initial state
    after each test.
    """
    # Make the database writeable, as we are going to modify it
    store_path, store_cache = _store_dir_and_cache
    _recursive_chmod(store_path, 0o755)

    yield database_mutable_config

    # Restore the initial state by copying the content of the cache back into
    # the store and making the database read-only
    shutil.rmtree(store_path)
    copy_tree(str(store_cache), str(store_path))
    _recursive_chmod(store_path, 0o555)


@pytest.fixture()
def dirs_with_libfiles(tmp_path_factory: pytest.TempPathFactory):
    lib_to_libfiles = {
        "libstdc++": ["libstdc++.so", "libstdc++.tbd"],
        "libgfortran": ["libgfortran.a", "libgfortran.dylib"],
        "libirc": ["libirc.a", "libirc.so"],
    }

    root = tmp_path_factory.mktemp("root")
    lib_to_dirs = {}
    i = 0
    for lib, libfiles in lib_to_libfiles.items():
        dirs = []
        for libfile in libfiles:
            lib_dir = root / str(i)
            lib_dir.mkdir()
            (lib_dir / libfile).touch()
            dirs.append(str(lib_dir))
            i += 1
        lib_to_dirs[lib] = dirs

    all_dirs = list(itertools.chain.from_iterable(lib_to_dirs.values()))

    yield lib_to_dirs, all_dirs


def _return_none(*args):
    return None


@pytest.fixture(autouse=True)
def disable_compiler_output_cache(monkeypatch):
    monkeypatch.setattr(
        spack.compilers.libraries, "COMPILER_CACHE", spack.compilers.libraries.CompilerCache()
    )


@pytest.fixture(scope="function")
def install_mockery(temporary_store: spack.store.Store, mutable_config, mock_packages):
    """Hooks a fake install directory, DB, and stage directory into Spack."""
    # We use a fake package, so temporarily disable checksumming
    with spack.config.override("config:checksum", False):
        yield

    # Wipe out any cached prefix failure locks (associated with the session-scoped mock archive)
    temporary_store.failure_tracker.clear_all()


@pytest.fixture(scope="function")
def temporary_mirror(mutable_config, tmp_path_factory):
    mirror_dir = tmp_path_factory.mktemp("mirror")
    mirror_cmd("add", "test-mirror-func", mirror_dir.as_uri())
    yield str(mirror_dir)


@pytest.fixture(scope="function")
def temporary_store(tmp_path: Path, request):
    """Hooks a temporary empty store for the test function."""
    ensure_configuration_fixture_run_before(request)
    temporary_store_path = tmp_path / "opt"
    with spack.store.use_store(str(temporary_store_path)) as s:
        yield s
    if temporary_store_path.exists():
        shutil.rmtree(temporary_store_path)


@pytest.fixture()
def mock_fetch(mock_archive, monkeypatch):
    """Fake the URL for a package so it downloads from a file."""
    monkeypatch.setattr(
        spack.package_base.PackageBase, "fetcher", URLFetchStrategy(url=mock_archive.url)
    )


class MockResourceFetcherGenerator:
    def __init__(self, url):
        self.url = url

    def _generate_fetchers(self, *args, **kwargs):
        return [URLFetchStrategy(url=self.url)]


@pytest.fixture()
def mock_resource_fetch(mock_archive, monkeypatch):
    """Fake fetcher generator that works with resource stages to redirect to a file."""
    mfg = MockResourceFetcherGenerator(mock_archive.url)
    monkeypatch.setattr(spack.stage.ResourceStage, "_generate_fetchers", mfg._generate_fetchers)


class MockLayout:
    def __init__(self, root):
        self.root = root

    def path_for_spec(self, spec):
        return os.path.sep.join([self.root, spec.name + "-" + spec.dag_hash()])

    def ensure_installed(self, spec):
        pass


@pytest.fixture()
def gen_mock_layout(tmp_path: Path):
    # Generate a MockLayout in a temporary directory. In general the prefixes
    # specified by MockLayout should never be written to, but this ensures
    # that even if they are, that it causes no harm
    def create_layout(root):
        subroot = tmp_path / root
        subroot.mkdir(parents=True, exist_ok=True)
        return MockLayout(str(subroot))

    yield create_layout


class MockConfig:
    def __init__(self, configuration, writer_key):
        self._configuration = configuration
        self.writer_key = writer_key

    def configuration(self, module_set_name):
        return self._configuration

    def writer_configuration(self, module_set_name):
        return self.configuration(module_set_name)[self.writer_key]


class ConfigUpdate:
    def __init__(self, root_for_conf, writer_mod, writer_key, monkeypatch):
        self.root_for_conf = root_for_conf
        self.writer_mod = writer_mod
        self.writer_key = writer_key
        self.monkeypatch = monkeypatch

    def __call__(self, filename):
        file = os.path.join(self.root_for_conf, filename + ".yaml")
        with open(file, encoding="utf-8") as f:
            config_settings = syaml.load_config(f)
        spack.config.set("modules:default", config_settings)
        mock_config = MockConfig(config_settings, self.writer_key)

        self.monkeypatch.setattr(spack.modules.common, "configuration", mock_config.configuration)
        self.monkeypatch.setattr(
            self.writer_mod, "configuration", mock_config.writer_configuration
        )
        self.monkeypatch.setattr(self.writer_mod, "configuration_registry", {})


@pytest.fixture()
def module_configuration(monkeypatch, request, mutable_config):
    """Reads the module configuration file from the mock ones prepared
    for tests and monkeypatches the right classes to hook it in.
    """
    # Class of the module file writer
    writer_cls = getattr(request.module, "writer_cls")
    # Module where the module file writer is defined
    writer_mod = inspect.getmodule(writer_cls)
    # Key for specific settings relative to this module type
    writer_key = str(writer_mod.__name__).split(".")[-1]
    # Root folder for configuration
    root_for_conf = os.path.join(spack.paths.test_path, "data", "modules", writer_key)

    # ConfigUpdate, when called, will modify configuration, so we need to use
    # the mutable_config fixture
    return ConfigUpdate(root_for_conf, writer_mod, writer_key, monkeypatch)


@pytest.fixture()
def mock_gnupghome(monkeypatch):
    # GNU PGP can't handle paths longer than 108 characters (wtf!@#$) so we
    # have to make our own tmp_path with a shorter name than pytest's.
    # This comes up because tmp paths on macOS are already long-ish, and
    # pytest makes them longer.
    try:
        spack.util.gpg.init()
    except spack.util.gpg.SpackGPGError:
        if not spack.util.gpg.GPG:
            pytest.skip("This test requires gpg")

    short_name_tmpdir = tempfile.mkdtemp()
    with spack.util.gpg.gnupghome_override(short_name_tmpdir):
        yield short_name_tmpdir

    # clean up, since we are doing this manually
    # Ignore errors cause we seem to be hitting a bug similar to
    # https://bugs.python.org/issue29699 in CI (FileNotFoundError: [Errno 2] No such
    # file or directory: 'S.gpg-agent.extra').
    shutil.rmtree(short_name_tmpdir, ignore_errors=True)


##########
# Fake archives and repositories
##########


@pytest.fixture(scope="session", params=[(".tar.gz", "z")])
def mock_archive(request, tmp_path_factory: pytest.TempPathFactory):
    """Creates a very simple archive directory with a configure script and a
    makefile that installs to a prefix. Tars it up into an archive.
    """
    try:
        tar = spack.util.executable.which("tar", required=True)
    except spack.util.executable.CommandNotFoundError:
        pytest.skip("requires tar to be installed")

    tmpdir = tmp_path_factory.mktemp("mock-archive-dir")
    source_dir = tmpdir / spack.stage._source_path_subdir
    source_dir.mkdir()
    repodir = source_dir

    # Create the configure script
    configure_path = str(source_dir / "configure")
    with open(configure_path, "w", encoding="utf-8") as f:
        f.write(
            "#!/bin/sh\n"
            "prefix=$(echo $1 | sed 's/--prefix=//')\n"
            "cat > Makefile <<EOF\n"
            "all:\n"
            "\techo Building...\n\n"
            "install:\n"
            "\tmkdir -p $prefix\n"
            "\ttouch $prefix/dummy_file\n"
            "EOF\n"
        )
    os.chmod(configure_path, 0o755)

    # Archive it
    with working_dir(str(tmpdir)):
        archive_name = "{0}{1}".format(spack.stage._source_path_subdir, request.param[0])
        tar("-c{0}f".format(request.param[1]), archive_name, spack.stage._source_path_subdir)

    Archive = collections.namedtuple(
        "Archive", ["url", "path", "archive_file", "expanded_archive_basedir"]
    )
    archive_file = str(tmpdir / archive_name)
    url = url_util.path_to_file_url(archive_file)

    # Return the url
    yield Archive(
        url=url,
        archive_file=archive_file,
        path=str(repodir),
        expanded_archive_basedir=spack.stage._source_path_subdir,
    )


def _parse_cvs_date(line):
    """Turn a CVS log date into a datetime.datetime"""
    # dates in CVS logs can have slashes or dashes and may omit the time zone:
    # date: 2021-07-07 02:43:33 -0700;  ...
    # date: 2021-07-07 02:43:33;  ...
    # date: 2021/07/07 02:43:33;  ...
    m = re.search(r"date:\s+(\d+)[/-](\d+)[/-](\d+)\s+(\d+):(\d+):(\d+)", line)
    if not m:
        return None
    year, month, day, hour, minute, second = [int(g) for g in m.groups()]
    return datetime.datetime(year, month, day, hour, minute, second)


@pytest.fixture(scope="session")
def mock_cvs_repository(tmp_path_factory: pytest.TempPathFactory):
    """Creates a very simple CVS repository with two commits and a branch."""
    cvs = spack.util.executable.which("cvs", required=True)

    tmpdir = tmp_path_factory.mktemp("mock-cvs-repo-dir")
    source_dir = tmpdir / spack.stage._source_path_subdir
    source_dir.mkdir()
    repodir = source_dir
    cvsroot = str(repodir)

    # The CVS repository and source tree need to live in a different directories
    sourcedirparent = tmp_path_factory.mktemp("mock-cvs-source-dir")
    module = spack.stage._source_path_subdir
    url = cvsroot + "%module=" + module
    source_module_dir = sourcedirparent / module
    source_module_dir.mkdir()
    sourcedir = source_module_dir

    def format_date(date):
        if date is None:
            return None
        return date.strftime("%Y-%m-%d %H:%M:%S")

    def get_cvs_timestamp(output):
        """Find the most recent CVS time stamp in a `cvs log` output"""
        latest_timestamp = None
        for line in output.splitlines():
            timestamp = _parse_cvs_date(line)
            if timestamp:
                if latest_timestamp is None:
                    latest_timestamp = timestamp
                else:
                    latest_timestamp = max(latest_timestamp, timestamp)
        return latest_timestamp

    # We use this to record the time stamps for when we create CVS revisions,
    # so that we can later check that we retrieve the proper commits when
    # specifying a date. (CVS guarantees checking out the lastest revision
    # before or on the specified date). As we create each revision, we
    # separately record the time by querying CVS.
    revision_date = {}

    # Initialize the repository
    with working_dir(str(sourcedir)):
        cvs("-d", cvsroot, "init")
        cvs(
            "-d",
            cvsroot,
            "import",
            "-m",
            "initial mock repo commit",
            module,
            "mockvendor",
            "mockrelease",
        )
        with working_dir(str(sourcedirparent)):
            cvs("-d", cvsroot, "checkout", module)

        # Commit file r0
        r0_file = "r0_file"
        (sourcedir / r0_file).touch()
        cvs("-d", cvsroot, "add", r0_file)
        cvs("-d", cvsroot, "commit", "-m", "revision 0", r0_file)
        output = cvs("log", "-N", r0_file, output=str)
        revision_date["1.1"] = format_date(get_cvs_timestamp(output))

        # Commit file r1
        r1_file = "r1_file"
        (sourcedir / r1_file).touch()
        cvs("-d", cvsroot, "add", r1_file)
        cvs("-d", cvsroot, "commit", "-m", "revision 1", r1_file)
        output = cvs("log", "-N", r0_file, output=str)
        revision_date["1.2"] = format_date(get_cvs_timestamp(output))

        # Create branch 'mock-branch'
        cvs("-d", cvsroot, "tag", "mock-branch-root")
        cvs("-d", cvsroot, "tag", "-b", "mock-branch")

    # CVS does not have the notion of a unique branch; branches and revisions
    # are managed separately for every file
    def get_branch():
        """Return the branch name if all files are on the same branch, else
        return None. Also return None if all files are on the trunk."""
        lines = cvs("-d", cvsroot, "status", "-v", output=str).splitlines()
        branch = None
        for line in lines:
            m = re.search(r"(\S+)\s+[(]branch:", line)
            if m:
                tag = m.group(1)
                if branch is None:
                    # First branch name found
                    branch = tag
                elif tag == branch:
                    # Later branch name found; all branch names found so far
                    # agree
                    pass
                else:
                    # Later branch name found; branch names differ
                    branch = None
                    break
        return branch

    # CVS does not have the notion of a unique revision; usually, one uses
    # commit dates instead
    def get_date():
        """Return latest date of the revisions of all files"""
        output = cvs("log", "-N", r0_file, output=str)
        timestamp = get_cvs_timestamp(output)
        if timestamp is None:
            return None
        return format_date(timestamp)

    checks = {
        "default": Bunch(file=r1_file, branch=None, date=None, args={"cvs": url}),
        "branch": Bunch(
            file=r1_file,
            branch="mock-branch",
            date=None,
            args={"cvs": url, "branch": "mock-branch"},
        ),
        "date": Bunch(
            file=r0_file,
            branch=None,
            date=revision_date["1.1"],
            args={"cvs": url, "date": revision_date["1.1"]},
        ),
    }

    test = Bunch(
        checks=checks, url=url, get_branch=get_branch, get_date=get_date, path=str(repodir)
    )

    yield test


@pytest.fixture(scope="session")
def mock_git_repository(git, tmp_path_factory: pytest.TempPathFactory):
    """Creates a git repository multiple commits, branches, submodules, and
    a tag. Visual representation of the commit history (starting with the
    earliest commit at c0)::

       c3       c1 (test-branch, r1)  c2 (tag-branch)
        |______/_____________________/
       c0 (r0)

    We force the default branch to be "main" to ensure that it behaves with package class tests.

    There are two branches aside from 'default': 'test-branch' and 'tag-branch';
    each has one commit; the tag-branch has a tag referring to its commit
    (c2 in the diagram).

    Two submodules are added as part of the very first commit on 'default'; each
    of these refers to a repository with a single commit.

    c0, c1, and c2 include information to define explicit versions in the
    associated builtin_mock package 'git-test'. c3 is a commit in the
    repository but does not have an associated explicit package version.
    """
    suburls = []
    # Create two git repositories which will be used as submodules in the
    # main repository
    for submodule_count in range(2):
        tmpdir = tmp_path_factory.mktemp("mock-git-repo-submodule-dir-{0}".format(submodule_count))
        source_dir = tmpdir / spack.stage._source_path_subdir
        source_dir.mkdir()
        repodir = source_dir
        suburls.append((submodule_count, url_util.path_to_file_url(str(repodir))))

        with working_dir(str(repodir)):
            git("init")
            git("config", "user.name", "Spack")
            git("config", "user.email", "spack@spack.io")

            # r0 is just the first commit
            submodule_file = "r0_file_{0}".format(submodule_count)
            (repodir / submodule_file).touch()
            git("add", submodule_file)
            git(
                "-c",
                "commit.gpgsign=false",
                "commit",
                "-m",
                "mock-git-repo r0 {0}".format(submodule_count),
            )

    tmpdir = tmp_path_factory.mktemp("mock-git-repo-dir")
    source_dir = tmpdir / spack.stage._source_path_subdir
    source_dir.mkdir()
    repodir = source_dir

    # Create the main repository
    with working_dir(str(repodir)):
        git("init")
        git("config", "user.name", "Spack")
        git("config", "user.email", "spack@spack.io")
        git("checkout", "-b", "main")
        url = url_util.path_to_file_url(str(repodir))
        for number, suburl in suburls:
            git("submodule", "add", suburl, "third_party/submodule{0}".format(number))

        # r0 is the first commit: it consists of one file and two submodules
        r0_file = "r0_file"
        (repodir / r0_file).touch()
        git("add", r0_file)
        git("-c", "commit.gpgsign=false", "commit", "-m", "mock-git-repo r0")

        branch = "test-branch"
        branch_file = "branch_file"
        git("branch", branch)

        tag_branch = "tag-branch"
        tag_file = "tag_file"
        git("branch", tag_branch)

        # Check out test branch and add one commit
        git("checkout", branch)
        (repodir / branch_file).touch()
        git("add", branch_file)
        git("-c", "commit.gpgsign=false", "commit", "-m", "r1 test branch")

        # Check out the tag branch, add one commit, and then add a tag for it
        git("checkout", tag_branch)
        (repodir / tag_file).touch()
        git("add", tag_file)
        git("-c", "commit.gpgsign=false", "commit", "-m", "tag test branch")

        tag = "test-tag"
        git("tag", tag)

        default_branch = "main"
        git("checkout", default_branch)

        r2_file = "r2_file"
        (repodir / r2_file).touch()
        git("add", r2_file)
        git("-c", "commit.gpgsign=false", "commit", "-m", "mock-git-repo r2")

        rev_hash = lambda x: git("rev-parse", x, output=str).strip()
        r2 = rev_hash(default_branch)

        # annotated tag
        a_tag = "annotated-tag"
        git("tag", "-a", a_tag, "-m", "annotated tag")

        # Record the commit hash of the (only) commit from test-branch and
        # the file added by that commit
        r1 = rev_hash(branch)
        r1_file = branch_file

        multiple_directories_branch = "many_dirs"
        num_dirs = 3
        num_files = 2
        dir_files = []
        for i in range(num_dirs):
            for j in range(num_files):
                dir_files.append(f"dir{i}/file{j}")

        git("checkout", "-b", multiple_directories_branch)
        for f in dir_files:
            file_path = repodir / f
            file_path.parent.mkdir(parents=True, exist_ok=True)
            file_path.touch()
            git("add", f)

        git("-c", "commit.gpgsign=false", "commit", "-m", "many_dirs add files")

        # restore default
        git("checkout", default_branch)

    # Map of version -> bunch. Each bunch includes; all the args
    # that must be specified as part of a version() declaration (used to
    # manufacture a version for the 'git-test' package); the associated
    # revision for the version; a file associated with (and particular to)
    # that revision/branch.
    checks = {
        "default": Bunch(revision=default_branch, file=r0_file, args={"git": url}),
        "branch": Bunch(revision=branch, file=branch_file, args={"git": url, "branch": branch}),
        "tag-branch": Bunch(
            revision=tag_branch, file=tag_file, args={"git": url, "branch": tag_branch}
        ),
        "tag": Bunch(revision=tag, file=tag_file, args={"git": url, "tag": tag}),
        "commit": Bunch(revision=r1, file=r1_file, args={"git": url, "commit": r1}),
        "annotated-tag": Bunch(revision=a_tag, file=r2_file, args={"git": url, "tag": a_tag}),
        # In this case, the version() args do not include a 'git' key:
        # this is the norm for packages, so this tests how the fetching logic
        # would most-commonly assemble a Git fetcher
        "default-no-per-version-git": Bunch(
            revision=default_branch, file=r0_file, args={"branch": default_branch}
        ),
        "many-directories": Bunch(
            revision=multiple_directories_branch,
            file=dir_files[0],
            args={"git": url, "branch": multiple_directories_branch},
        ),
    }

    t = Bunch(
        checks=checks,
        url=url,
        hash=rev_hash,
        path=str(repodir),
        git_exe=git,
        unversioned_commit=r2,
    )
    yield t


@pytest.fixture(scope="function")
def mock_git_test_package(mock_git_repository, mutable_mock_repo, monkeypatch):
    # install a fake git version in the package class
    pkg_class = spack.repo.PATH.get_pkg_class("git-test")
    monkeypatch.delattr(pkg_class, "git")
    monkeypatch.setitem(pkg_class.versions, spack.version.Version("git"), mock_git_repository.url)
    return pkg_class


@pytest.fixture(scope="session")
def mock_hg_repository(tmp_path_factory: pytest.TempPathFactory):
    """Creates a very simple hg repository with two commits."""
    try:
        hg = spack.util.executable.which("hg", required=True)
    except spack.util.executable.CommandNotFoundError:
        pytest.skip("requires mercurial to be installed")

    tmpdir = tmp_path_factory.mktemp("mock-hg-repo-dir")
    source_dir = tmpdir / spack.stage._source_path_subdir
    source_dir.mkdir()
    repodir = source_dir

    get_rev = lambda: hg("id", "-i", output=str).strip()

    # Initialize the repository
    with working_dir(str(repodir)):
        url = url_util.path_to_file_url(str(repodir))
        hg("init")

        # Commit file r0
        r0_file = "r0_file"
        (repodir / r0_file).touch()
        hg("add", r0_file)
        hg("commit", "-m", "revision 0", "-u", "test")
        r0 = get_rev()

        # Commit file r1
        r1_file = "r1_file"
        (repodir / r1_file).touch()
        hg("add", r1_file)
        hg("commit", "-m", "revision 1", "-u", "test")
        r1 = get_rev()

    checks = {
        "default": Bunch(revision=r1, file=r1_file, args={"hg": str(repodir)}),
        "rev0": Bunch(revision=r0, file=r0_file, args={"hg": str(repodir), "revision": r0}),
    }
    t = Bunch(checks=checks, url=url, hash=get_rev, path=str(repodir))
    yield t


@pytest.fixture(scope="session")
def mock_svn_repository(tmp_path_factory: pytest.TempPathFactory):
    """Creates a very simple svn repository with two commits."""
    try:
        svn = spack.util.executable.which("svn", required=True)
        svnadmin = spack.util.executable.which("svnadmin", required=True)
    except spack.util.executable.CommandNotFoundError:
        pytest.skip("requires svn to be installed")

    tmpdir = tmp_path_factory.mktemp("mock-svn-stage")
    source_dir = tmpdir / spack.stage._source_path_subdir
    source_dir.mkdir()
    repodir = source_dir
    url = url_util.path_to_file_url(str(repodir))

    # Initialize the repository
    with working_dir(str(repodir)):
        # NOTE: Adding --pre-1.5-compatible works for NERSC
        # Unknown if this is also an issue at other sites.
        svnadmin("create", "--pre-1.5-compatible", str(repodir))

        # Import a structure (first commit)
        r0_file = "r0_file"
        tmp_path = tmpdir / "tmp-path"
        tmp_path.mkdir()
        (tmp_path / r0_file).touch()
        svn("import", str(tmp_path), url, "-m", "Initial import r0")
        shutil.rmtree(tmp_path, onerror=onerror)

        # Second commit
        r1_file = "r1_file"
        svn("checkout", url, str(tmp_path))
        (tmp_path / r1_file).touch()

        with working_dir(str(tmp_path)):
            svn("add", str(tmp_path / r1_file))
            svn("ci", "-m", "second revision r1")

        shutil.rmtree(tmp_path, onerror=onerror)
        r0 = "1"
        r1 = "2"

    checks = {
        "default": Bunch(revision=r1, file=r1_file, args={"svn": url}),
        "rev0": Bunch(revision=r0, file=r0_file, args={"svn": url, "revision": r0}),
    }

    def get_rev():
        output = svn("info", "--xml", output=str)
        info = xml.etree.ElementTree.fromstring(output)
        return info.find("entry/commit").get("revision")

    t = Bunch(checks=checks, url=url, hash=get_rev, path=str(repodir))
    yield t


@pytest.fixture(scope="function")
def mutable_mock_env_path(tmp_path: Path, mutable_config, monkeypatch):
    """Fixture for mocking the internal spack environments directory."""
    mock_path = tmp_path / "mock-env-path"
    mutable_config.set("config:environments_root", str(mock_path))
    monkeypatch.setattr(ev.environment, "default_env_path", str(mock_path))
    return mock_path


@pytest.fixture()
def installation_dir_with_headers(tmp_path_factory: pytest.TempPathFactory):
    """Mock installation tree with a few headers placed in different
    subdirectories. Shouldn't be modified by tests as it is session
    scoped.
    """
    root = tmp_path_factory.mktemp("prefix")

    # Create a few header files:
    #
    # <prefix>
    # |-- include
    # |   |--boost
    # |   |   |-- ex3.h
    # |   |-- ex3.h
    # |-- path
    #     |-- to
    #         |-- ex1.h
    #         |-- subdir
    #             |-- ex2.h
    #
    (root / "include" / "boost").mkdir(parents=True)
    (root / "include" / "boost" / "ex3.h").touch()
    (root / "include" / "ex3.h").touch()
    (root / "path" / "to").mkdir(parents=True)
    (root / "path" / "to" / "ex1.h").touch()
    (root / "path" / "to" / "subdir").mkdir()
    (root / "path" / "to" / "subdir" / "ex2.h").touch()

    return root


##########
# Specs of various kind
##########


@pytest.fixture(params=["conflict+foo%clang", "conflict-parent@0.9^conflict~foo"])
def conflict_spec(request):
    """Specs which violate constraints specified with the "conflicts"
    directive in the "conflict" package.
    """
    return request.param


@pytest.fixture(scope="module")
def mock_test_repo(tmp_path_factory: pytest.TempPathFactory):
    """Create an empty repository."""
    repo_namespace = "mock_test_repo"
    repodir = tmp_path_factory.mktemp(repo_namespace)
    packages_dir = repodir / spack.repo.packages_dir_name
    packages_dir.mkdir()
    yaml_path = repodir / "repo.yaml"
    yaml_path.write_text(
        """
repo:
    namespace: mock_test_repo
"""
    )

    with spack.repo.use_repositories(str(repodir)) as repo:
        yield repo, repodir

    shutil.rmtree(str(repodir))


@pytest.fixture(scope="function")
def mock_clone_repo(tmp_path_factory: pytest.TempPathFactory):
    """Create a cloned repository."""
    repo_namespace = "mock_clone_repo"
    repodir = tmp_path_factory.mktemp(repo_namespace)
    yaml_path = repodir / "repo.yaml"
    yaml_path.write_text(
        """
repo:
    namespace: mock_clone_repo
"""
    )

    shutil.copytree(
        os.path.join(spack.paths.mock_packages_path, spack.repo.packages_dir_name),
        os.path.join(str(repodir), spack.repo.packages_dir_name),
    )

    with spack.repo.use_repositories(str(repodir)) as repo:
        yield repo, repodir

    shutil.rmtree(str(repodir))


##########
# Class and fixture to work around problems raising exceptions in directives,
# which cause tests like test_from_list_url to hang for Python 2.x metaclass
# processing.
#
# At this point only version and patch directive handling has been addressed.
##########


class MockBundle:
    has_code = False
    name = "mock-bundle"


@pytest.fixture
def mock_directive_bundle():
    """Return a mock bundle package for directive tests."""
    return MockBundle()


@pytest.fixture
def clear_directive_functions():
    """Clear all overidden directive functions for subsequent tests."""
    yield

    # Make sure any directive functions overidden by tests are cleared before
    # proceeding with subsequent tests that may depend on the original
    # functions.
    spack.directives_meta.DirectiveMeta._directives_to_be_executed = []


@pytest.fixture
def mock_executable(tmp_path: Path):
    """Factory to create a mock executable in a temporary directory that
    output a custom string when run.
    """
    shebang = "#!/bin/sh\n" if sys.platform != "win32" else "@ECHO OFF\n"

    def _factory(name, output, subdir=("bin",)):
        executable_dir = tmp_path.joinpath(*subdir)
        executable_dir.mkdir(parents=True, exist_ok=True)
        executable_path = executable_dir / name
        if sys.platform == "win32":
            executable_path = executable_dir / (name + ".bat")
        executable_path.write_text(f"{shebang}{output}\n")
        executable_path.chmod(0o755)
        return executable_path

    return _factory


@pytest.fixture()
def mock_test_stage(mutable_config, tmp_path: Path):
    # NOTE: This fixture MUST be applied after any fixture that uses
    # the config fixture under the hood
    # No need to unset because we use mutable_config
    tmp_stage = str(tmp_path / "test_stage")
    mutable_config.set("config:test_stage", tmp_stage)

    yield tmp_stage


@pytest.fixture(autouse=True)
def inode_cache():
    spack.llnl.util.lock.FILE_TRACKER.purge()
    yield
    # TODO: it is a bug when the file tracker is non-empty after a test,
    # since it means a lock was not released, or the inode was not purged
    # when acquiring the lock failed. So, we could assert that here, but
    # currently there are too many issues to fix, so look for the more
    # serious issue of having a closed file descriptor in the cache.
    assert not any(f.fh.closed for f in spack.llnl.util.lock.FILE_TRACKER._descriptors.values())
    spack.llnl.util.lock.FILE_TRACKER.purge()


@pytest.fixture(autouse=True)
def brand_new_binary_cache():
    yield
    spack.binary_distribution.BINARY_INDEX = spack.llnl.util.lang.Singleton(
        spack.binary_distribution.BinaryCacheIndex
    )


def _trivial_package_hash(spec: spack.spec.Spec) -> str:
    """Return a trivial package hash for tests to avoid expensive AST parsing."""
    # Pad package name to consistent length and cap at 32 chars for realistic hash length
    return base64.b32encode(f"{spec.name:<32}".encode()[:32]).decode().lower()


@pytest.fixture(autouse=True)
def mock_package_hash_for_tests(request, monkeypatch):
    """Replace expensive package hash computation with trivial one for tests.
    Tests can force the real package hash by using the @pytest.mark.use_package_hash marker."""
    if "use_package_hash" in request.keywords:
        yield
        return
    pkg_hash = spack.hash_types.package_hash
    idx = spack.hash_types.HASHES.index(pkg_hash)
    mock_pkg_hash = spack.hash_types.SpecHashDescriptor(
        depflag=0, package_hash=True, name="package_hash", override=_trivial_package_hash
    )
    monkeypatch.setattr(spack.hash_types, "package_hash", mock_pkg_hash)
    try:
        spack.hash_types.HASHES[idx] = mock_pkg_hash
        yield
    finally:
        spack.hash_types.HASHES[idx] = pkg_hash


@pytest.fixture()
def noncyclical_dir_structure(tmp_path: Path):
    """
    Create some non-trivial directory structure with
    symlinks to dirs and dangling symlinks, but no cycles::

        .
        |-- a/
        |   |-- d/
        |   |-- file_1
        |   |-- to_file_1 -> file_1
        |   `-- to_c -> ../c
        |-- b -> a
        |-- c/
        |   |-- dangling_link -> nowhere
        |   `-- file_2
        `-- file_3
    """
    d = tmp_path / "nontrivial-dir"
    d.mkdir()
    j = os.path.join

    with working_dir(str(d)):
        os.mkdir(j("a"))
        os.mkdir(j("a", "d"))
        with open(j("a", "file_1"), "wb"):
            pass
        os.symlink(j("file_1"), j("a", "to_file_1"))
        os.symlink(j("..", "c"), j("a", "to_c"))
        os.symlink(j("a"), j("b"))
        os.mkdir(j("c"))
        os.symlink(j("nowhere"), j("c", "dangling_link"))
        with open(j("c", "file_2"), "wb"):
            pass
        with open(j("file_3"), "wb"):
            pass
    yield d


@pytest.fixture(scope="function")
def mock_config_data():
    config_data_dir = os.path.join(spack.paths.test_path, "data", "config")
    return config_data_dir, os.listdir(config_data_dir)


@pytest.fixture(scope="function")
def mock_curl_configs(mock_config_data, monkeypatch):
    """
    Mock curl-based retrieval of configuration files from the web by grabbing
    them from the test data configuration directory.

    Fetches a single (configuration) file if the name matches one in the test
    data directory.
    """
    config_data_dir, config_files = mock_config_data

    class MockCurl:
        def __init__(self):
            self.returncode = None

        def __call__(self, *args, **kwargs):
            url = [a for a in args if a.startswith("http")][0]
            basename = os.path.basename(url)
            if os.path.splitext(url)[1]:
                if basename in config_files:
                    filename = os.path.join(config_data_dir, basename)

                    with open(filename, "r", encoding="utf-8") as f:
                        lines = f.readlines()
                        write_file(os.path.basename(filename), "".join(lines))

                    self.returncode = 0
                else:
                    # This is a "404" and is technically only returned if -f
                    # flag is provided to curl.
                    tty.msg("curl: (22) The requested URL returned error: 404")
                    self.returncode = 22

    monkeypatch.setattr(spack.util.web, "require_curl", MockCurl)


@pytest.fixture(scope="function")
def mock_fetch_url_text(mock_config_data, monkeypatch):
    """Mock spack.util.web.fetch_url_text."""

    stage_dir, config_files = mock_config_data

    def _fetch_text_file(url, dest_dir):
        raw_url = raw_github_gitlab_url(url)
        mkdirp(dest_dir)
        basename = os.path.basename(raw_url)
        src = join_path(stage_dir, basename)
        dest = join_path(dest_dir, basename)
        copy(src, dest)
        return dest

    monkeypatch.setattr(spack.util.web, "fetch_url_text", _fetch_text_file)


@pytest.fixture(scope="function")
def mock_tty_stdout(monkeypatch):
    """Make sys.stdout.isatty() return True, while forcing no color output."""
    monkeypatch.setattr(sys.stdout, "isatty", lambda: True)
    with spack.llnl.util.tty.color.color_when("never"):
        yield


@pytest.fixture
def prefix_like():
    return "package-0.0.0.a1-hashhashhashhashhashhashhashhash"


@pytest.fixture()
def prefix_tmpdir(tmp_path: Path, prefix_like: str):
    prefix_dir = tmp_path / prefix_like
    prefix_dir.mkdir()
    return prefix_dir


@pytest.fixture()
def binary_with_rpaths(prefix_tmpdir: Path):
    """Factory fixture that compiles an ELF binary setting its RPATH. Relative
    paths are encoded with `$ORIGIN` prepended.
    """

    def _factory(rpaths, message="Hello world!", dynamic_linker="/lib64/ld-linux.so.2"):
        source = prefix_tmpdir / "main.c"
        source.write_text(
            """
        #include <stdio.h>
        int main(){{
            printf("{0}");
        }}
        """.format(
                message
            )
        )
        gcc = spack.util.executable.which("gcc", required=True)
        executable = source.parent / "main.x"
        # Encode relative RPATHs using `$ORIGIN` as the root prefix
        rpaths = [x if os.path.isabs(x) else os.path.join("$ORIGIN", x) for x in rpaths]
        opts = [
            "-Wl,--disable-new-dtags",
            f"-Wl,-rpath={':'.join(rpaths)}",
            f"-Wl,--dynamic-linker,{dynamic_linker}",
            str(source),
            "-o",
            str(executable),
        ]
        gcc(*opts)
        return executable

    return _factory


@pytest.fixture(scope="session")
def concretized_specs_cache():
    """Cache for mock concrete specs"""
    return {}


@pytest.fixture
def default_mock_concretization(config, mock_packages, concretized_specs_cache):
    """Return the default mock concretization of a spec literal, obtained using the mock
    repository and the mock configuration.

    This fixture is unsafe to call in a test when either the default configuration or mock
    repository are not used or have been modified.
    """

    def _func(spec_str, tests=False):
        key = spec_str, tests
        if key not in concretized_specs_cache:
            concretized_specs_cache[key] = spack.concretize.concretize_one(
                spack.spec.Spec(spec_str), tests=tests
            )
        return concretized_specs_cache[key].copy()

    return _func


@pytest.fixture
def shell_as(shell):
    if sys.platform != "win32":
        yield
        return
    if shell not in ("pwsh", "bat"):
        raise RuntimeError("Shell must be one of supported Windows shells (pwsh|bat)")
    try:
        # fetch and store old shell type
        _shell = os.environ.get("SPACK_SHELL", None)
        os.environ["SPACK_SHELL"] = shell
        yield
    finally:
        # restore old shell if one was set
        if _shell:
            os.environ["SPACK_SHELL"] = _shell


@pytest.fixture()
def nullify_globals(request, monkeypatch):
    ensure_configuration_fixture_run_before(request)
    monkeypatch.setattr(spack.config, "CONFIG", None)
    monkeypatch.setattr(spack.caches, "MISC_CACHE", None)
    monkeypatch.setattr(spack.caches, "FETCH_CACHE", None)
    monkeypatch.setattr(spack.repo, "PATH", None)
    monkeypatch.setattr(spack.store, "STORE", None)


def pytest_runtest_setup(item):
    # Skip test marked "not_on_windows" if they're run on Windows
    not_on_windows_marker = item.get_closest_marker(name="not_on_windows")
    if not_on_windows_marker and sys.platform == "win32":
        pytest.skip(*not_on_windows_marker.args)

    # Skip items marked "only windows" if they're run anywhere but Windows
    only_windows_marker = item.get_closest_marker(name="only_windows")
    if only_windows_marker and sys.platform != "win32":
        pytest.skip(*only_windows_marker.args)


@pytest.fixture(autouse=True)
def disable_parallelism(monkeypatch, request):
    """Disable process pools in tests. Enabled by default to avoid oversubscription when running
    under pytest-xdist. Can be overridden with `@pytest.mark.enable_parallelism`."""
    if "enable_parallelism" not in request.keywords:
        monkeypatch.setattr(spack.util.parallel, "ENABLE_PARALLELISM", False)


def _root_path(x, y, *, path):
    return path


@pytest.fixture
def mock_modules_root(tmp_path: Path, monkeypatch):
    """Sets the modules root to a temporary directory, to avoid polluting configuration scopes."""
    fn = functools.partial(_root_path, path=str(tmp_path))
    monkeypatch.setattr(spack.modules.common, "root_path", fn)


@pytest.fixture()
def compiler_factory():
    """Factory for a compiler dict, taking a spec and an OS as arguments."""

    def _factory(*, spec):
        return {
            "spec": f"{spec}",
            "prefix": "/path",
            "extra_attributes": {"compilers": {"c": "/path/bin/cc", "cxx": "/path/bin/cxx"}},
        }

    return _factory


@pytest.fixture()
def host_architecture_str():
    """Returns the broad architecture family (x86_64, aarch64, etc.)"""
    return str(spack.vendor.archspec.cpu.host().family)


def _true(x):
    return True


def _libc_from_python(self):
    return spack.spec.Spec("glibc@=2.28", external_path="/some/path")


@pytest.fixture()
def do_not_check_runtimes_on_reuse(monkeypatch):
    monkeypatch.setattr(spack.solver.reuse, "_has_runtime_dependencies", _true)


@pytest.fixture(autouse=True, scope="session")
def _c_compiler_always_exists():
    fn = spack.solver.asp.c_compiler_runs
    spack.solver.asp.c_compiler_runs = _true
    mthd = spack.compilers.libraries.CompilerPropertyDetector.default_libc
    spack.compilers.libraries.CompilerPropertyDetector.default_libc = _libc_from_python
    yield
    spack.solver.asp.c_compiler_runs = fn
    spack.compilers.libraries.CompilerPropertyDetector.default_libc = mthd


@pytest.fixture(scope="session")
def mock_test_cache(tmp_path_factory: pytest.TempPathFactory):
    cache_dir = tmp_path_factory.mktemp("cache")
    return spack.util.file_cache.FileCache(cache_dir)


class MockHTTPResponse(io.IOBase):
    """This is a mock HTTP response, which implements part of http.client.HTTPResponse"""

    def __init__(self, status, reason, headers=None, body=None):
        self.msg = None
        self.version = 11
        self.url = None
        self.headers = email.message.EmailMessage()
        self.status = status
        self.code = status
        self.reason = reason
        self.debuglevel = 0
        self._body = body

        if headers is not None:
            for key, value in headers.items():
                self.headers[key] = value

    @classmethod
    def with_json(cls, status, reason, headers=None, body=None):
        """Create a mock HTTP response with JSON string as body"""
        body = io.BytesIO(json.dumps(body).encode("utf-8"))
        return cls(status, reason, headers, body)

    def read(self, *args, **kwargs):
        return self._body.read(*args, **kwargs)

    def getheader(self, name, default=None):
        self.headers.get(name, default)

    def getheaders(self):
        return self.headers.items()

    def fileno(self):
        return 0

    def getcode(self):
        return self.status

    def info(self):
        return self.headers


@pytest.fixture()
def mock_runtimes(config, mock_packages):
    return mock_packages.packages_with_tags("runtime")


@pytest.fixture()
def write_config_file(tmp_path: Path):
    """Returns a function that writes a config file."""

    def _write(config, data, scope):
        config_dir = tmp_path / scope
        config_dir.mkdir(parents=True, exist_ok=True)
        config_yaml = config_dir / (config + ".yaml")
        with config_yaml.open("w") as f:
            syaml.dump_config(data, f)
        return config_yaml

    return _write


def _include_cache_root():
    return join_path(str(tempfile.mkdtemp()), "user_cache", "includes")


@pytest.fixture()
def mock_include_cache(monkeypatch):
    """Override the include cache directory so tests don't pollute user cache."""
    monkeypatch.setattr(spack.config, "_include_cache_location", _include_cache_root)


@pytest.fixture()
def wrapper_dir(install_mockery):
    """Installs the compiler wrapper and returns the prefix where the script is installed."""
    wrapper = spack.concretize.concretize_one("compiler-wrapper")
    wrapper_pkg = wrapper.package
    PackageInstaller([wrapper_pkg], explicit=True).install()
    return wrapper_pkg.bin_dir()


def _noop(*args, **kwargs):
    pass


@pytest.fixture(autouse=True)
def no_compilers_init(monkeypatch):
    """Disables automatic compiler initialization"""
    monkeypatch.setattr(spack.compilers.config, "_init_packages_yaml", _noop)


@pytest.fixture(autouse=True)
def skip_provenance_check(monkeypatch, request):
    """Skip binary provenance check for git versions

    Binary provenance checks require querying git repositories and mirrors.
    The infrastructure for this is complex and a heavy lift for simple things like spec syntax
    checks. This fixture defaults to skipping this check, but can be overridden with the
    @pytest.mark.require_provenance decorator
    """
    if "require_provenance" not in request.keywords:
        monkeypatch.setattr(spack.package_base.PackageBase, "_resolve_git_provenance", _noop)


@pytest.fixture(scope="function")
def config_two_gccs(mutable_config):
    # Configure two gcc compilers that could be concretized to
    extra_attributes_block = {
        "compilers": {"c": "/path/to/gcc", "cxx": "/path/to/g++", "fortran": "/path/to/fortran"}
    }
    mutable_config.set(
        "packages:gcc:externals::",
        [
            {
                "spec": "gcc@12.3.1 languages=c,c++,fortran",
                "prefix": "/path",
                "extra_attributes": extra_attributes_block,
            },
            {
                "spec": "gcc@10.3.1 languages=c,c++,fortran",
                "prefix": "/path",
                "extra_attributes": extra_attributes_block,
            },
        ],
    )


@pytest.fixture(scope="function")
def mock_util_executable(monkeypatch):
    logger = []
    should_fail = []
    registered_reponses = {}

    def mock_call(self, *args, **kwargs):
        cmd = self.exe + list(args)
        str_cmd = " ".join(map(str, cmd))
        logger.append(str_cmd)
        for failure_key in should_fail:
            if failure_key in str_cmd:
                self.returncode = 1
                if kwargs.get("fail_on_error", True):
                    raise spack.util.executable.ProcessError(f"Failed: {str_cmd}")
                return
        for key, value in registered_reponses.items():
            if key in str_cmd:
                return value
        self.returncode = 0

    monkeypatch.setattr(spack.util.executable.Executable, "__call__", mock_call)
    yield logger, should_fail, registered_reponses


@pytest.fixture()
def reset_extension_paths():
    """Clears the cache used for entry points, both in setup and tear-down.
    Needed if a test stresses parts related to computing paths for Spack extensions
    """
    spack.extensions.extension_paths_from_entry_points.cache_clear()
    yield
    spack.extensions.extension_paths_from_entry_points.cache_clear()
